import React, { useState, useEffect, useRef } from 'react';
import { Plus, X, ChevronLeft, ChevronRight, ChevronDown, Copy, Edit2, Trash2, Check, GripVertical, Menu, Palette } from 'lucide-react';
import { Reorder } from 'framer-motion';

export function SheetTabs({ sheets, activeSheetId, onSwitch, onAdd, onRename, onDelete, onDuplicate, onReorder, onColorChange }) {
    const [editingId, setEditingId] = useState(null);
    const [editName, setEditName] = useState("");
    const [menuState, setMenuState] = useState({ id: null, x: 0, y: 0 }); // Changed to store position
    const scrollContainerRef = useRef(null);
    const menuRef = useRef(null);

    // Initial load scrolling
    useEffect(() => {
        if (activeSheetId && scrollContainerRef.current) {
            const activeTab = scrollContainerRef.current.querySelector(`[data-sheet-id="${activeSheetId}"]`);
            if (activeTab) {
                activeTab.scrollIntoView({ behavior: 'smooth', block: 'nearest', inline: 'center' });
            }
        }
    }, [activeSheetId]);

    // Close menu on outside click or scroll
    useEffect(() => {
        const handleClickOutside = (event) => {
            if (menuRef.current && !menuRef.current.contains(event.target)) {
                setMenuState({ id: null, x: 0, y: 0 });
            }
        };
        const handleScroll = () => {
            if (menuState.id) setMenuState({ id: null, x: 0, y: 0 });
        };

        document.addEventListener('mousedown', handleClickOutside);
        window.addEventListener('resize', handleScroll); // Close on resize too
        // Attach scroll listener to the scroll container to close menu on scroll
        const scrollContainer = scrollContainerRef.current;
        if (scrollContainer) scrollContainer.addEventListener('scroll', handleScroll);

        return () => {
            document.removeEventListener('mousedown', handleClickOutside);
            window.removeEventListener('resize', handleScroll);
            if (scrollContainer) scrollContainer.removeEventListener('scroll', handleScroll);
        };
    }, [menuState.id]);

    const handleStartEdit = (sheet) => {
        setEditingId(sheet.id);
        setEditName(sheet.name);
        setMenuState({ id: null, x: 0, y: 0 });
    };

    const handleFinishEdit = () => {
        if (editingId && editName.trim()) {
            onRename(editingId, editName.trim());
        }
        setEditingId(null);
        setEditName("");
    };

    const handleKeyDown = (e) => {
        if (e.key === 'Enter') handleFinishEdit();
        if (e.key === 'Escape') setEditingId(null);
    };

    const handleMenuOpen = (e, sheetId) => {
        e.preventDefault();
        e.stopPropagation();
        const rect = e.currentTarget.getBoundingClientRect();
        // Position menu above the tab, aligned left
        setMenuState({
            id: sheetId,
            x: rect.left,
            y: rect.top - 10 // 10px buffer above tab
        });
    };

    const scroll = (direction) => {
        if (scrollContainerRef.current) {
            const scrollAmount = 200;
            scrollContainerRef.current.scrollBy({
                left: direction === 'left' ? -scrollAmount : scrollAmount,
                behavior: 'smooth'
            });
        }
    };

    const handleColorSelect = (color) => {
        if (menuState.id && onColorChange) {
            onColorChange(menuState.id, color);
            setMenuState({ id: null, x: 0, y: 0 });
        }
    };

    const COLORS = [
        '#ef4444', '#f97316', '#f59e0b', '#84cc16', '#10b981',
        '#06b6d4', '#3b82f6', '#8b5cf6', '#d946ef', '#f43f5e', '#64748b'
    ];

    return (
        <div className="sheet-bar-container glass-panel-bottom">

            {/* Add Button & All Sheets Menu */}
            <div className="sheet-control-group glass-control">
                <button onClick={onAdd} className="sheet-icon-btn" title="Add Sheet">
                    <Plus size={20} strokeWidth={2} />
                </button>
                <button className="sheet-icon-btn" title="All Sheets">
                    <Menu size={18} strokeWidth={2} />
                </button>
            </div>

            {/* Scroll Left */}
            <button onClick={() => scroll('left')} className="scroll-btn left md:flex hidden glass-control">
                <ChevronLeft size={16} />
            </button>

            {/* Scrollable Tabs Area */}
            <div className="sheet-scroll-area custom-scrollbar" ref={scrollContainerRef}>
                <Reorder.Group
                    as="div"
                    axis="x"
                    values={sheets}
                    onReorder={onReorder || (() => { })}
                    className="flex flex-row h-full items-end"
                    style={{ display: 'flex', flexDirection: 'row' }}
                >
                    {sheets.map(sheet => {
                        const isActive = sheet.id === activeSheetId;
                        const isEditing = sheet.id === editingId;
                        const isMenuOpen = sheet.id === menuState.id;

                        return (
                            <Reorder.Item
                                key={sheet.id}
                                value={sheet}
                                id={sheet.id}
                                className="h-full flex items-end outline-none"
                                whileDrag={{ scale: 1.05, opacity: 0.8 }}
                                style={{ display: 'flex', flexDirection: 'row' }}
                            >
                                <div
                                    data-sheet-id={sheet.id}
                                    onClick={() => !isEditing && onSwitch(sheet.id)}
                                    onDoubleClick={() => handleStartEdit(sheet)}
                                    onContextMenu={(e) => handleMenuOpen(e, sheet.id)}
                                    className={`sheet-tab glass-tab ${isActive ? 'active' : ''} ${isMenuOpen ? 'menu-open' : ''}`}
                                    style={{
                                        borderBottomColor: isActive ? '#10b981' : (sheet.color || 'transparent')
                                    }}
                                >
                                    {/* Color Indicator (if active or colored) */}
                                    {sheet.color && !isActive && (
                                        <div className="absolute bottom-0 left-0 w-full h-[3px]" style={{ backgroundColor: sheet.color }}></div>
                                    )}

                                    {/* Sheet Name */}
                                    <div className="sheet-tab-content">
                                        {isEditing ? (
                                            <input
                                                autoFocus
                                                type="text"
                                                value={editName}
                                                onChange={(e) => setEditName(e.target.value)}
                                                onBlur={handleFinishEdit}
                                                onKeyDown={handleKeyDown}
                                                className="sheet-tab-input"
                                                onClick={(e) => e.stopPropagation()}
                                            />
                                        ) : (
                                            <span style={{ color: isActive ? '#10b981' : 'inherit' }}>{sheet.name}</span>
                                        )}
                                    </div>

                                    {/* Dropdown Trigger */}
                                    <button
                                        onClick={(e) => handleMenuOpen(e, sheet.id)}
                                        className="sheet-menu-trigger"
                                    >
                                        <ChevronDown size={12} strokeWidth={2.5} />
                                    </button>
                                </div>
                            </Reorder.Item>
                        );
                    })}
                </Reorder.Group>
            </div>

            {/* Scroll Right */}
            <button onClick={() => scroll('right')} className="scroll-btn md:flex hidden glass-control">
                <ChevronRight size={16} />
            </button>

            {/* Global Fixed Position Menu */}
            {menuState.id && (
                <div
                    ref={menuRef}
                    className="sheet-context-menu glass-menu fixed"
                    style={{
                        left: menuState.x,
                        top: 'auto',
                        bottom: window.innerHeight - menuState.y + 5, // Position above the tab
                        position: 'fixed',
                        zIndex: 9999
                    }}
                    onClick={(e) => e.stopPropagation()}
                >
                    <button
                        className="menu-item"
                        onClick={() => {
                            if (sheets.length > 1 && window.confirm(`Delete ${sheet.name}?`)) onDelete(menuState.id);
                            setMenuState({ id: null, x: 0, y: 0 });
                        }}
                        disabled={sheets.length <= 1}
                    >
                        <Trash2 size={14} className="mr-2 text-rose-500" />
                        <span>Delete</span>
                    </button>
                    <button
                        className="menu-item"
                        onClick={() => {
                            onDuplicate && onDuplicate(menuState.id);
                            setMenuState({ id: null, x: 0, y: 0 });
                        }}
                    >
                        <Copy size={14} className="mr-2 text-blue-500" />
                        Duplicate
                    </button>
                    <button
                        className="menu-item"
                        onClick={() => {
                            const sheet = sheets.find(s => s.id === menuState.id);
                            if (sheet) handleStartEdit(sheet);
                        }}
                    >
                        <Edit2 size={14} className="mr-2 text-slate-500" />
                        Rename
                    </button>

                    <div className="menu-divider"></div>

                    <div className="p-2">
                        <div className="text-xs text-slate-500 mb-2 px-2 flex items-center gap-1">
                            <Palette size={12} /> Tab Color
                        </div>
                        <div className="grid grid-cols-6 gap-1 px-2">
                            <button onClick={() => handleColorSelect(null)} className="w-5 h-5 rounded-full border border-slate-200 hover:scale-110 transition-transform relative" title="None">
                                <div className="absolute inset-0 m-auto w-[1px] h-full bg-slate-400 rotate-45"></div>
                            </button>
                            {COLORS.map(c => (
                                <button
                                    key={c}
                                    onClick={() => handleColorSelect(c)}
                                    className="w-5 h-5 rounded-full hover:scale-110 transition-transform"
                                    style={{ backgroundColor: c }}
                                />
                            ))}
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
}
